/*
 * Autor: Jörg Weule ( weule@7b5.de )
 * ©2017 Germany
 *
 * Lizenz: GNU GENERAL PUBLIC LICENSE - Version 3, 29 June 2007
 *
 */

#define _FILE_OFFSET_BITS 64
#define _LARGEFILE_SOURCE 1
#define _LARGEFILE64_SOURCE 1

#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <time.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <utime.h>

#define PID_13BIT(a,b) (((( (unsigned int)a)&0x1f)<<8) | (unsigned int)b )
#define LEN_12BIT(a,b) (((( (unsigned int)a)&0x0f)<<8) | (unsigned int)b )
#define PGM_16BIT(a,b) (((  (unsigned int)a      )<<8) | (unsigned int)b )

typedef unsigned char byte;
static  unsigned int  crc_table[256]; // Table of 8-bit remainders

#define BUFSIZE (1024*1024)
#define MAXTAB 16
byte blk[188];
byte pat[188];
byte pmt[188];
byte sit[188 * MAXTAB];
byte tab[188 * MAXTAB];

long long totalCount = 0 ;

int pat_found = 0;
int sit_found = 0;
int sit_count = 0;

char* thisPgName = "unknown";
int thisPG = 0;
int pmtPID = 0;
FILE* logFile = NULL;

#define MAXPIDS (1<<13)
int pid2pg[MAXPIDS]; // pid -> sid

byte* iso8895_9[256] = {
    /*   1   2    3    4   5   6   7    8   9   A      B    C    D    E   F */
    " ", " ", " ", " ", " ", " ", " ", " ", " ", " ", " ", " ", " ", "\n", " ", " ",
    " ", " ", " ", " ", " ", " ", " ", " ", " ", " ", " ", " ", " ", " ", " ", " ",
    " ", "!", "\"", "#", "$", "%", "&", "'", "(", ")", "*", "+", ",", "-", ".", "/",
    "0", "1", "2", "3", "4", "5", "6", "7", "8", "9", ":", ";", "<", "=", ">", "?",
    "@", "A", "B", "C", "D", "E", "F", "G", "H", "I", "J", "K", "L", "M", "N", "O",
    "P", "Q", "R", "S", "T", "U", "V", "W", "X", "Y", "Z", "[", "\\", "]", "^", "_",
    "`", "a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o",
    "p", "q", "r", "s", "t", "u", "v", "w", "x", "y", "z", "{", "|", "}", "~", " ",
    " ", " ", " ", " ", " ", " ", " ", " ", " ", " ", "\n\n", " ", " ", " ", " ", " ",
    " ", " ", " ", " ", " ", " ", " ", " ", " ", " ", " ", " ", " ", " ", " ", " ",
    " ", "¡", "¢", "£", "¤", "¥", "¦", "§", "¨", "©", "ª", "«", "¬", " ", "®", "¯",
    "°", "±", "²", "³", "´", "µ", "¶", "·", "¸", "¹", "º", "»", "¼", "½", "¾", "¿",
    "À", "Á", "Â", "Ã", "Ä", "Å", "Æ", "Ç", "È", "É", "Ê", "Ë", "Ì", "Í", "Î", "Ï",
    "Ğ", "Ñ", "Ò", "Ó", "Ô", "Õ", "Ö", "×", "Ø", "Ù", "Ú", "Û", "Ü", "İ", "Ş", "ß",
    "à", "á", "â", "ã", "ä", "å", "æ", "ç", "è", "é", "ê", "ë", "ì", "í", "î", "ï",
    "ğ", "ñ", "ò", "ó", "ô", "õ", "ö", "÷", "ø", "ù", "ú", "û", "ü", "ı", "ş", "ÿ"
};

void gen_crc_table(void) {
    unsigned int i, j, k;
    unsigned int x;
    for (i = 0; i < 256; i++) {
        k = i << 24 ;
        for (j = 0; j < 8; j++) {
            if (k & 0x80000000 )
                k = (k << 1) ^ 0x04c11db7;
            else
                k = (k << 1);
        }
        crc_table[i] = k ;
    }
}

int crc32(byte *data_blk_ptr, int data_blk_size) {
    unsigned int i, j, k = 0xffffffff ;
    for (j = 0; j < data_blk_size; j++) {
        i = ((k >> 24) ^ *data_blk_ptr++) & 0xff ;
        k = (k << 8) ^ crc_table[i];
    }
    return k;
}

void init_tabs() {
    int i;
    for (i = 0; i < MAXPIDS; i++) {
        pid2pg[i] = 0;
    }
    gen_crc_table();
    memset(pat,188,1);
    memset(pmt,188,1);
    byte* ibuf = malloc(BUFSIZE);
    byte* obuf = malloc(BUFSIZE);
}

void bufcpy(byte*tgt, byte*src, int l) {
    while (l--) {
        if (*src < 128) {
            *tgt++ = *src++;
        } else {
            byte *iso = iso8895_9[*src];
            if (iso != NULL) {
                while (*iso) {
                    *tgt++ = *iso++;
                }
                src++;
            } else {
                *tgt++ = *src++;
            }
        }
    }
    *tgt = '\0';
}

void copyText(byte *tgt, byte *src) {
    int len = src[0];
    if (len == 0) return;

    byte * s = tgt + strlen(tgt);
    if (src[1] >= 32) {
        bufcpy(s, src + 1, len);
    } else {
        if ( src[1] != 5 ) {
            fprintf(logFile,"Encoding: %02x\n", src[1]);
        }
        bufcpy(s, src + 2, len - 1);
    }
}

int getTab(
        byte *ts_block_array,
        int ts_block_count,
        byte *ts_new_block
) {

    int pid = PID_13BIT(ts_block_array[1],ts_block_array[2]);
    int i = 0;
    int tab_bytes = 0 ;

    {
        memset(tab, 0, sizeof(tab));
        memcpy(tab, ts_block_array, 188);
        tab_bytes = 188 ;
    }
    
    if ( ( tab[3] & 0x20 ) != 0 ) {
        int l = tab[4] + 1 ;
        memcpy(tab + 4, tab + 4 + l, 188 - 4 - l);
        tab[3] = tab[3] & 0xdf ;
        tab_bytes -= l ;
    }

    {
        int l = tab[4];
        if ( l > 0 ) {
            memcpy(tab + 4, tab + 4 + l, tab_bytes - l - 4 );
            tab[4] = '\0';
            tab_bytes -= l ;
        }
    }

    int tab_num = tab[5];
    int tab_end = LEN_12BIT(tab[6],tab[7]) + 8 ;

    // concatenate the payload data and check sequence numbers
    int n = ts_block_array[3] & 0xf;
    for (i = 1; i < ts_block_count; i++) {
        n = (n + 1) & 0xf;
        int n2 = ts_block_array[188 * i + 3] & 0xf;
        if (n2 != n) {
            fprintf(logFile,"WARN: Missing block no %1x\n",n);
            return 0;
        }
        memcpy(tab + tab_bytes, ts_block_array + i * 188 + 4, 184);
        tab_bytes += 184 ;
    }

    // concatenate the end of the block if at the new start block
    if ( tab_bytes < tab_end ) {
        if ( ts_new_block != NULL && (ts_new_block[1] & 0x40) != 0 ) {
            n = (n + 1) & 0xf;
            int n2 = ts_new_block[3] & 0xf;
            int l = ts_new_block[4];
            if (n2 != n && l > 0 ) {
                fprintf(logFile,"WARN: Missing block no %1x\n",n);
                return 0;
            }
            if (l > 0) {
                memcpy(tab + tab_bytes, ts_new_block + 5, l);
                tab_bytes += l ;
            }
            memset(tab+tab_bytes,0,188);
        }
    }
    
    if ( tab_bytes < tab_end ) {
        return 0 ;
    } else {
        int crc32result = crc32(tab+5,tab_end-5);
        if ( crc32result != 0 ) {
            fprintf(logFile,"ERROR: CRC32: %08x\n",crc32result);
            return 0 ;
        }
        return tab_end ;
    }

}

void clear_pid2pg(){
    int i ;
    for ( i = 0 ; i < MAXPIDS ; i++ ) pid2pg[i] = 0;
}

void map_pg2pmt(int pg,int pid){
    if ( thisPG == pg ) {
        if ( pmtPID == 0 ) fprintf(logFile,"INFO: PMT %d\n",pid);
        pmtPID = pid ;
        pid2pg[pid]=pg;
    }
}

void procPAT() {
    clear_pid2pg();

    int tabEnd = getTab(pat,1,NULL);
    if ( tabEnd == 0 ) {
        return;
    }

    int noc = (pat[7] - 9) / 4;
    int i;
    for (i = 0; i < noc; i++) {
        byte *p = pat + 13 + 4 * i;
        int tpg = PGM_16BIT(p[0], p[1]);
        int pmt_pid = PID_13BIT(p[2], p[3]);
        if (tpg != 0) {
            map_pg2pmt(tpg,pmt_pid);
        }
    }
    pat_found = 1;

}

void procPMT() {

    int tabEnd = getTab(blk,1,NULL);
    if ( tabEnd == 0 ) {
        return;
    }

    int pid = PID_13BIT(blk[1], blk[2]);

    int pg = thisPG;
    if (pg == 0) return;

    if (blk[5] != 0x02) return;

    int sl = blk[7];
    int pos = 17;
    while (pos < (sl + 4)) {
        int st = blk[pos];
        int epid = PID_13BIT(blk[pos + 1], blk[pos + 2]);
        if ( pid2pg[epid] == 0 ) {
            fprintf(logFile,"INFO:\tPID_%02x -> %d\n", st, epid);
        }
        pid2pg[epid] = pg;
        pos += 5 + blk[pos + 4];
    }
}

void procSIT() {
    if ((sit[1] & 0x40) == 0) {
        return;
    }

    int tabEnd = getTab(sit,sit_count,NULL);
    if ( tabEnd == 0 ) {
        return;
    }
    fprintf(logFile,"INFO: =============== %d %d\n",
        sit_count, tabEnd);
    
    int pid = PID_13BIT(sit[1], sit[2]);

    char*needle = thisPgName ;

    tabEnd -= 4 ;
    int pos1 = 16;
    while (pos1 < tabEnd) {
        int pos2 = pos1;
        int sit_pg = PGM_16BIT(tab[pos2], tab[pos2 + 1]);
        pos2 += 5;
        while ( tab[pos2] != 0x48 ) {
            if ( tab[pos2] == 0x50 ) {
                pos2 += 2 + tab[pos2+1] ;
            } else {
                pos2 += 2 + tab[pos2+1] ;
            }
        }

        // decode descriptor 0x48
        pos2 = pos2 + 3 ;
        // skip the text at pos 8 (len-byte + text)
        int len = tab[pos2];
        pos2 += 1 + len;
        len = tab[pos2];

        byte *pg_name = malloc(len * 2);
        *pg_name = '\0' ;
        copyText(pg_name,tab+pos2);

        // skip the program name (len-byte + text)
        pos2 += 1 + len;

        // skip to next channel with pos1
        len = tab[pos1 + 4] + 5;
        pos1 = pos1 + len;

        if ( 0 == strcmp(pg_name,needle) ) {
            thisPG = sit_pg;
            thisPgName = pg_name;
        } else {
            if ( thisPG == 0 && NULL != strstr(pg_name,needle) ) {
                thisPG = sit_pg;
                thisPgName = pg_name;
            }
        }
        fprintf(logFile,"INFO: Program %5d: %s\n", sit_pg, pg_name);
    }
    if ( thisPG == 0 ) {
        fprintf(logFile,"ERROR: Program '%s' not found.\n",needle);
    }
    fprintf(logFile,"INFO: Using   %5d -> '%s'\n", thisPG, thisPgName);
    if ( thisPG != 0 ) {
        sit_found = 1;
    }
    fprintf(logFile,"\n");
}

int readBlk(){
    int n = fread(blk, 1, 188, stdin);
    while ( 1 ){
        if ( n <= 0 ) return 0;
        if (blk[0] != 0x47) {
            fprintf(logFile,"WARN: Broken package stream at %lld.\n",totalCount);
            while (0 < ( n = fread(blk, 1, 1, stdin))) {
                if ( blk[0] == 0x47 ) break;
            }
            if ( n <= 0 ) return 0;
            if ( 187 >= ( n = fread(blk+1, 1, 187, stdin))){
                fprintf(logFile,"WARN: Broken package stream, end at %lld.\n",totalCount);
                return 0;
            }
            fprintf(logFile,"INFO: Continuing at %lld.\n",totalCount);
            continue; // dont trust this block
        }
        if ((blk[1] & 0x80) != 0) {
            fprintf(logFile,"WARN: Broken package.\n");
            continue;
        }
        return 188;
    }
    return 188;
}

int main(int argc, char**argv) {

    int fileNum = 0;

    for ( int i = 1 ; i < argc ; i++ ) {
        if (0 == strcmp(argv[i], "-l")||0 == strcmp(argv[i],"--log")) {
            logFile = fopen(argv[++i],"w");
        }
        if (0 == strcmp(argv[i], "-n")||0 == strcmp(argv[i],"--name")) {
            thisPgName = argv[++i];
        }
        if (0 == strcmp(argv[i], "--help")) {
            printf("Usage: %s "
                "[-l|--log logfile] "
                "[-n|--name ChannelName]"
                " \n", argv[0]);
            return 0 ;
        }
    }

    if ( logFile == NULL ) {
        logFile = stderr ;
    }

    if ( thisPgName == NULL ) {
        fprintf(logFile,"WARN: No program name.\n");
    }

    init_tabs();

    while (188 == readBlk()) {
        totalCount += 188 ;

        int pid = PID_13BIT(blk[1], blk[2]);
        /* count pid packages */
        int pg = pid2pg[pid];

        if (pid == 0 && pat_found == 0) {
            if ( 0 != memcmp(pat,blk,188) ) {
                memcpy(pat, blk, 188);
                procPAT();
            }
        }
        if (pid == 17 && sit_found == 0) {
            if ((blk[1] & 0x40) != 0) {
                sit_count = 0;
            }
            if ( sit_count == MAXTAB ) {
                fprintf(logFile,"ERROR: SIT table bigger than expected.\n");
                return 1;
            }
            memcpy(sit + 188 * sit_count, blk, 188);
            sit_count++;
            if ( (sit[1] & 0x40) != 0 ) {
                procSIT();
                if ( pat_found ) procPAT();
            }
        }
        if (pid == pmtPID) {
            if ( 0 != memcmp(pmt,blk,188) ) {
                memcpy(pmt, blk, 188);
                procPMT();
            }
        }

        /*
         * write program streams
         */
        if (pid == 0 || pid == 17 || pid == 20 || pid2pg[pid] != 0 ) {
            fwrite(blk, 188, 1, stdout);
        }

    }

    fprintf(logFile,"INFO: Bytes read: %lld\n", totalCount);
    fflush(logFile);
    fflush(stdout);
    return 0;

}

